/** \file threshmodel.h
 * **  \A model which receives an analog input and converts it into
 * **  \threshold output
 * **  \Implementation file for the class
 * **  \ (c), 2004, Prashant Joshi (joshi@igi.tugraz.at)
 * */

#include "threshmodel.h"
#include <math.h>
#include <mat.h>

/** Constructs a new threshold model. This model has 2 inputs and 2 outputs. */

ThreshModel::ThreshModel() : PhysicalModel(2, 2)
{
    // First you have to register the names of your input- and output-channels
    // (for the connect command)
    register_input_channel("ro_lo");  // index 0
    register_input_channel("ro_hi");  // index 1
    
    // Register the 2 output channels
    char outputname[10];
    sprintf(outputname, "thresh_lo");
    register_output_channel(outputname);
    sprintf(outputname, "thresh_hi");
    register_output_channel(outputname);
    
    //! The time-step of simulation.
    model_DT = 0.002; 
    TIMESTEPS = 0;
    cur_timestep = 0;
    nr_file_steps = (int) (model_DT / 1e-4);
    sim_step_counter = 0;
    
    // Create an output-buffer
    buffer_length = 2;
    buffer_position = 0;
    out_buffer = new (double *)[2];
    for (int i=0; i<2; i++)
        out_buffer[i] = new double[buffer_length];
}

/** Frees the memory. */
ThreshModel::~ThreshModel() 
{
    if (out_buffer)
    {
        for (int i=0; i<2; i++)
        {
            delete[] out_buffer[i];
        }
        delete[] out_buffer;
    }
    out_buffer = 0;
}

/** Transforms the current inputs to new output values.
    \param I Array of pointers to input values from the readouts.
    \param O Array of output values.
    \return -1 if an error occured, 1 for success. */
int ThreshModel::transform(double** I, double* O) 
{
    int i;
    // Watch if a time-step change occurs
    if (++sim_step_counter >= nr_file_steps)
    {
        cur_timestep++;
        sim_step_counter = 0;
    }
    if (cur_timestep < 1)
    {
        return 1;
    }
    
    // Get input values
    double ns_ip1 = *(I[0]);
    double ns_ip2 = *(I[1]);
    
    // Calculate next position in circular buffer
    int next_buffer_position = (buffer_position+1) % buffer_length; 
    out_buffer[1][next_buffer_position] = (ns_ip1>=0)?0.5:-0.5;
    out_buffer[2][next_buffer_position] = (ns_ip2>=0)?0.5:-0.5;
    
    //Write output
    for (i=0; i<2; i++)
    {
        O[i] = out_buffer[i][buffer_position];
    }
    
    // NO DELAY!!!
    buffer_position = next_buffer_position;
    return 1;
}

/** This function is called after parameters are updated. */
int ThreshModel::updateInternal()
{
    nr_file_steps = (int) (model_DT / DT);
    return 0;
}
